package com.droideve.apps.dealify.activities;

import android.content.Intent;
import android.os.Bundle;
import android.util.Log;
import android.view.MenuItem;
import android.view.View;
import android.widget.Button;
import android.widget.LinearLayout;
import android.widget.TextView;
import android.widget.Toast;

import androidx.annotation.Nullable;
import androidx.appcompat.widget.Toolbar;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;
import androidx.swiperefreshlayout.widget.SwipeRefreshLayout;

import com.android.volley.DefaultRetryPolicy;
import com.android.volley.Request;
import com.android.volley.RequestQueue;
import com.android.volley.Response;
import com.android.volley.VolleyError;
import com.droideve.apps.dealify.R;
import com.droideve.apps.dealify.Services.BusStation;
import com.droideve.apps.dealify.Services.GenericNotifyEvent;
import com.droideve.apps.dealify.adapter.lists.CategoriesListAdapter;
import com.droideve.apps.dealify.adapter.order.OrdersListExpandAdapter;
import com.droideve.apps.dealify.animation.LineItemDecoration;
import com.droideve.apps.dealify.appconfig.Constances;
import com.droideve.apps.dealify.classes.Category;
import com.droideve.apps.dealify.classes.Order;
import com.droideve.apps.dealify.classes.User;
import com.droideve.apps.dealify.controllers.categories.CategoryController;
import com.droideve.apps.dealify.controllers.orders.OrdersController;
import com.droideve.apps.dealify.controllers.sessions.SessionsController;
import com.droideve.apps.dealify.load_manager.ViewManager;
import com.droideve.apps.dealify.network.ServiceHandler;
import com.droideve.apps.dealify.network.VolleySingleton;
import com.droideve.apps.dealify.network.api_request.ApiRequest;
import com.droideve.apps.dealify.network.api_request.ApiRequestListeners;
import com.droideve.apps.dealify.network.api_request.SimpleRequest;
import com.droideve.apps.dealify.parser.Parser;
import com.droideve.apps.dealify.parser.api_parser.OrderParser;
import com.droideve.apps.dealify.parser.tags.Tags;
import com.droideve.apps.dealify.utils.NSToast;

import org.greenrobot.eventbus.Subscribe;
import org.greenrobot.eventbus.ThreadMode;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import butterknife.BindView;
import butterknife.ButterKnife;
import io.realm.RealmList;

import static com.droideve.apps.dealify.appconfig.AppConfig.APP_DEBUG;


public class ListOrdersActivity extends GlobalActivity implements CategoriesListAdapter.ClickListener, SwipeRefreshLayout.OnRefreshListener, ViewManager.CustomView, OrdersListExpandAdapter.OnItemClickListener {

    //GET CATEGORIES FROM  DATABASE
    Toolbar toolbar;
    @BindView(R.id.toolbar_title)
    TextView toolbarTitle;
    @BindView(R.id.toolbar_subtitle)
    TextView toolbarDescription;
    @BindView(R.id.list)
    RecyclerView list;
    @BindView(R.id.refresh)
    SwipeRefreshLayout refresh;

    private OrdersListExpandAdapter mAdapter;
    private RequestQueue queue;
    private ViewManager mViewManager;

    //pager
    private int COUNT = 0;
    private int REQUEST_PAGE = 1;
    private int pastVisiblesItems, visibleItemCount, totalItemCount;
    private LinearLayoutManager mLayoutManager;
    private boolean loading = true;


    @Override
    protected void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        setContentView(R.layout.activity_categories);
        ButterKnife.bind(this);

        initToolbar();


        initComponent();


        setupRefresListener();


        setupViewManager();


        loadOrdersFromServer();


    }


    private void loadOrdersFromServer() {

        //load data from apis
        if (ServiceHandler.isNetworkAvailable(this)) {
            //open a specific order in the list
            if (getIntent().hasExtra("id")) {
                getOrdersFromApi(REQUEST_PAGE, getIntent().getExtras().getInt("id"));
            } else {
                getOrdersFromApi(REQUEST_PAGE, -1);
            }

        } else {
            refresh.setRefreshing(false);
            NSToast.show(getString(R.string.check_network));
            if (mAdapter.getItemCount() == 0)
                mViewManager.error();
        }
    }


    private void setupViewManager() {
        mViewManager = new ViewManager(this);
        mViewManager.setLoadingLayout(findViewById(R.id.loading));
        mViewManager.setResultLayout(findViewById(R.id.content));
        mViewManager.setErrorLayout(findViewById(R.id.error));
        mViewManager.setEmpty(findViewById(R.id.empty));
        mViewManager.setCustumizeView(this);
    }

    private void setupRefresListener() {
        refresh.setOnRefreshListener(this);
        refresh.setColorSchemeResources(
                R.color.colorAccent,
                R.color.colorAccent,
                R.color.colorAccent,
                R.color.colorAccent
        );
    }


    private void initComponent() {

        mLayoutManager = new LinearLayoutManager(this);
        list.setLayoutManager(mLayoutManager);
        list.addItemDecoration(new LineItemDecoration(this, LinearLayout.VERTICAL));
        list.setHasFixedSize(true);

        //set data and list adapter
        mAdapter = new OrdersListExpandAdapter(this, new ArrayList<>());
        list.setAdapter(mAdapter);

        // on item list clicked
        mAdapter.setOnItemClickListener(this);


        list.setOnScrollListener(new RecyclerView.OnScrollListener() {
            @Override
            public void onScrolled(RecyclerView recyclerView, int dx, int dy) {

                visibleItemCount = mLayoutManager.getChildCount();
                totalItemCount = mLayoutManager.getItemCount();
                pastVisiblesItems = mLayoutManager.findFirstVisibleItemPosition();

                if (loading) {

                    if ((visibleItemCount + pastVisiblesItems) >= totalItemCount) {
                        loading = false;

                        if (ServiceHandler.isNetworkAvailable(ListOrdersActivity.this)) {
                            if (COUNT > mAdapter.getItemCount())
                                getOrdersFromApi(REQUEST_PAGE, -1);
                        } else {
                            NSToast.show(getString( R.string.network_not_available));
                        }
                    }
                }
            }
        });


    }

    public List<Category> getData() {

        List<Category> results = new ArrayList<>();

        RealmList<Category> listCats = CategoryController.list();

        for (Category cat : listCats) {
            if (cat.getNumCat() > 0)
                results.add(cat);
        }


        return results;
    }


    // This method will be called when a Notification is posted (in the UI thread for Toast)
    @Subscribe(sticky = true, threadMode = ThreadMode.MAIN)
    public void onMessageEvent(GenericNotifyEvent event) {
        //refresh notification list when the product is deleted
        if (event.message != null && event.message.equals("order_updated")) {
            loadOrdersFromServer();
            event.message = null;
        }
    }

    @Override
    protected void onStart() {
        BusStation.getBus().register(this);
        super.onStart();
    }

    @Override
    protected void onPause() {
        super.onPause();
        BusStation.getBus().unregister(this);

    }

    @Override
    public void itemClicked(View view, final int position) {
    }

    @Override
    public boolean onOptionsItemSelected(MenuItem item) {

        if (android.R.id.home == item.getItemId()) {
            finish();
        }

        return super.onOptionsItemSelected(item);
    }

    @Override
    public void onBackPressed() {
        super.onBackPressed();

        

    }


    public void initToolbar() {

        toolbar = findViewById(R.id.app_bar);
        toolbarTitle.setText(R.string.orders);


        setSupportActionBar(toolbar);
        getSupportActionBar().setDisplayHomeAsUpEnabled(true);
        getSupportActionBar().setDisplayShowTitleEnabled(true);
        getSupportActionBar().setDisplayUseLogoEnabled(false);
        getSupportActionBar().setDisplayShowTitleEnabled(false);
        toolbarDescription.setVisibility(View.GONE);

    }


    @Override
    public void onRefresh() {
        loadOrdersFromServer();
    }


    private void getOrdersFromApi(final int page, final int order_id) {

        refresh.setRefreshing(true);
        queue = VolleySingleton.getInstance(this).getRequestQueue();

        //if not connected redirect to login page
        if (!SessionsController.isLogged()) {
            startActivity(new Intent(ListOrdersActivity.this, LoginV2Activity.class));
            
            finish();
            return;
        }

        final User mUser = SessionsController.getSession().getUser();

        final int user_id = mUser.getId();

        Map<String, String> params = new HashMap<String, String>();
        if (order_id > 0) params.put("order_id", String.valueOf(order_id));
        params.put("user_id", String.valueOf(user_id));
        params.put("page", String.valueOf(page));
        params.put("limit", "30");

        ApiRequest.newPostInstance(Constances.API.API_ORDERS_GET, new ApiRequestListeners() {
            @Override
            public void onSuccess(Parser parser) {

                refresh.setRefreshing(false);

                final OrderParser mCategoryParser = new OrderParser(parser);
                int success = Integer.parseInt(mCategoryParser.getStringAttr(Tags.SUCCESS));
                COUNT = 0;
                COUNT = mCategoryParser.getIntArg(Tags.COUNT);

                if (success == 0) {
                    return;
                }

                RealmList<Order> list = mCategoryParser.getOrders();

                if (page == 1) {
                    mAdapter.removeAll();
                }

                for (int i = 0; i < list.size(); i++) {
                    mAdapter.addItem(list.get(i));
                }

                if(REQUEST_PAGE == 1){
                    mAdapter.expandFirst();
                }

                //save data into the database
                if (list.size() > 0) {
                    OrdersController.insertOrders(list);
                }

                if (COUNT > mAdapter.getItemCount())
                    REQUEST_PAGE++;




            }

            @Override
            public void onFail(Map<String, String> errors) {
                refresh.setRefreshing(false);
            }
        }, params);


    }


    @Override
    public void customErrorView(View v) {

        Button retry = v.findViewById(R.id.btn);

        retry.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                getOrdersFromApi(REQUEST_PAGE, -1);
                REQUEST_PAGE = 1;
                mViewManager.loading();
            }
        });

    }

    @Override
    public void customLoadingView(View v) {
    }

    @Override
    public void customEmptyView(View v) {

        Button btn = v.findViewById(R.id.btn);
        btn.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                mViewManager.loading();
                getOrdersFromApi(1, -1);
                REQUEST_PAGE = 1;
            }
        });


    }

    @Override
    public void onItemClick(int position) {

    }

    @Override
    public void onOrderDetailClick(int position) {
        Intent intent = new Intent(this, OrderDetailActivity.class);
        intent.putExtra("id", mAdapter.getItem(position).getId());
        startActivity(intent);
    }


    @Override
    public void onOrderTrackClick(int position) {
        Intent intent = new Intent(this, DeliveryTrackingActivity.class);
        HashMap<String, String> params = new HashMap<>();
        params.put("delivery_id", String.valueOf(mAdapter.getItem(position).getDelivery_id()));
        intent.putExtra("params", params);

        startActivity(intent);

    }
}
